using System;
using System.Collections;
using System.Data;
using System.IO;
using System.Net;
using System.Net.Sockets;
using System.Reflection;
using System.Text;
using System.Threading;
using TABLE = gov.va.med.vbecs.Common.VbecsTables;
using ARTIFICIAL = gov.va.med.vbecs.Common.DatabaseConstants.ArtificialColumnNames;

namespace gov.va.med.vbecs.BOL
{
	/// <summary>
	/// Summary description for CautionTag.
	/// </summary>
	public class CautionTag : gov.va.med.vbecs.BOL.BaseBusinessObject
	{
		#region variables

		/// <summary>
		/// Blood unit associated with the caution tag
		/// </summary>
		private BOL.BloodUnit _bloodUnit;
		/// <summary>
		/// Patient associated with the caution tag
		/// </summary>
		private BOL.Patient _patient;
		/// <summary>
		/// VBECS user that is trying to print the tag
		/// </summary>
		private BOL.VbecsUser _user;
		private BOL.OrderedUnit _orderedUnit;
		/// <summary>
		/// Warning associated with printing the caution tag that will be displayed to the user
		/// </summary>
		private StringBuilder _message;
		/// <summary>
		/// Warning associated with printing the caution tag that will be displayed on the form
		/// </summary>
		private StringBuilder _formMessage;
		/// <summary>
		/// Warning messages sometimes require a comment in order to proceed with printing
		/// </summary>
		private String _overrideComment;
		/// <summary>
		/// Indicates whether or not the caution tag can be printed
		/// </summary>
		private bool _unitSelectable = true;
		/// <summary>
		/// Indicates whether or not override is required.
		/// </summary>
		private bool _overrideRequired = false;
		/// <summary>
		/// Inidicate whether unit must be emergency issued.
		/// </summary>
		private bool _emergencyIssue;
		/// <summary>
		/// Ordered component
		/// </summary>
		private BOL.OrderedComponent _orderedComponent;
		/// <summary>
		/// Indicates whether or not the current division is full service
		/// </summary>
		private bool _isFullService;
		/// <summary>
		/// Holds all entries for the exception report.
		/// </summary>
		private ArrayList _exceptionReportEntries;
		/// <summary>
		/// Only one instance of the Exception report need be generated.
		/// </summary>
		private bool _exceptionReportGenerated;
		/// <summary>
		/// Division
		/// </summary>
		private BOL.Division _division;
		/// <summary>
		/// Indicates whether an audio warning is needed.
		/// </summary>
		private bool _beep;
		/// <summary>
		/// This is the ZPL command for printing a blank tag
		/// CR3120: Updated ZPL to accomodate new format.
		/// </summary>
		private const string BLANK_TAG = "^XA^LH70,30^FO625,5,^FO20,40^A0N,32,25^FDProduct Code:^FS^FO20,230^A0N,56,39^FD^FS^FO20,250^A0N,56,39^FD^FS^FO1,440^GB1,725,4^FS^FO1,410^GB1080,725,4^FS^FO1,410^GB1080,0,4^FS^FO20,420^A0N,32,25^FDPatient Name:^FS^FO160,450^A0N,56,39^FD^FS^FO1,510^GB1080,0,4^FS^FO20,520^A0N,32,25^FDPatient ID:^FS^FO20,560^A0N,56,39^FD^FS^FO660,510^GB0,110,4^FS^FO690,520^A0N,32,25^FDRecipient Blood Type:^FS^FO690,560^A0N,56,39^FD^FS^FO1,620^GB1080,0,4^FS^FO20,630^A0N,32,25^FDUnit/Pool ID:^FS^FO20,670^A0N,56,39^FD^FS^FO660,620^GB0,115,4^FS^FO690,630^A0N,32,25^FDDonor Blood Type:^FS^FO690,670^A0N,56,39^FD^FS^FO1,730^GB1080,0,4^FS^FO20,760^A0N,32,25^FDCompatibility Interpretation:^FS^FO20,800^A0N,56,39^FD^FS^FO1,860^GB1080,0,4^FS^FO20,870^A0N,40,30^FDCAUTION: Before transfusing this blood component ^FS^FO640,870^A0N,40,30^FDperform bedside transfusion^FS^FO20,915^A0N,40,30^FDverification of the patient's identity by ^FS^FO485,915^A0N,40,30^FDmatching the patient identification on this^FS^FO20,960^A0N,40,30^FDtag with the patient wristband and the Blood ^FS^FO560,960^A0N,40,30^FDTransfusion Record Form.^FS^FO1,1030^GB1080,0,4^FS^FO20,1040^A0N,32,25^FDAssigned Date/Time:^FS^FO20,1080^A0N,56,39^FD^FS^FO430,1030^GB0,125,4^FS^FO480,1040^A0N,32,25^FDAssigning Tech/ Crossmatch Tech:^FS^FO480,1080^A0N,56,39^FD^FS^FO1,1155^GB1080,0,4^FS^XZ";

		/// <summary>
		/// Cache object for CR2899
		/// </summary>
		private Common.ABORhSource _patientABORhSourceCache;

		#endregion


		#region constructors

		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>4/27/2004</CreationDate>
		///<TestCases>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Constructor for printing blank tags/ forms only!  NOT for public consumption.
		/// </summary>
		private CautionTag()
		{
			_division = new BOL.Division(Common.LogonUser.LogonUserDivisionCode);
		}


		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>4/27/2004</CreationDate>
		///<TestCases>
		/// 
		/// <Case type="1" testid ="4146"> 
		///		<ExpectedInput>Valid blood unit, patient and user</ExpectedInput>
		///		<ExpectedOutput>Valid instance of CautionTag</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="4147"> 
		///		<ExpectedInput>Null blood unit</ExpectedInput>
		///		<ExpectedOutput>ArgumentException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Constructor for printing normal (non-blank) tags.  The rulesAlreadyValidated
		/// argument should only be set to true if the Caution Tag is being generated from 
		/// another UC that has already validated all business rules.   
		/// </summary>
		///
		public CautionTag(BOL.BloodUnit bloodUnit, 
			BOL.Patient patient, 
			BOL.VbecsUser user, 
			bool rulesAlreadyValidated): this(bloodUnit, 
								patient, 
								user, 
								rulesAlreadyValidated,
								null, null, null, null)
		{
			
		}

		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>4/27/2004</CreationDate>
		///<TestCases>
		/// 
		/// <Case type="1" testid ="4146"> 
		///		<ExpectedInput>Valid blood unit, patient and user</ExpectedInput>
		///		<ExpectedOutput>Valid instance of CautionTag</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="4147"> 
		///		<ExpectedInput>Null blood unit</ExpectedInput>
		///		<ExpectedOutput>ArgumentException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Constructor for printing normal (non-blank) tags.  The rulesAlreadyValidated
		/// argument should only be set to true if the Caution Tag is being generated from 
		/// another UC that has already validated all business rules.   
		/// </summary>
		/// 
		/// CR2899: Added the following arguments to the contructor: 
		///		OrderedComponent
		///		OrderedUnit
		///		Hashtable of antigen requirements
		///		Hashtable of component requirements
		///		
		///		These objects are optional and should only be used when doing bulk 
		///		processing.  Otherwise, pass in null.
		public CautionTag(BOL.BloodUnit bloodUnit, 
						BOL.Patient patient, 
						BOL.VbecsUser user, 
						bool rulesAlreadyValidated, 
						BOL.OrderedComponent orderedComponent, 
						BOL.OrderedUnit orderedUnit,
						Hashtable antigenNegsReqsCache,
						Hashtable componentsCache)
		{
			if (bloodUnit == null || patient == null || user == null)
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat("bloodunit, patient of user").ResString);
			}

			_bloodUnit = bloodUnit;
			_patient = patient;
			_user = user;


			// CR2899 start
			_patientABORhSourceCache = _patient.ABORhSource;

			if (orderedUnit == null)
			{
				_orderedUnit = new OrderedUnit(bloodUnit.BloodUnitGuid, patient.PatientGuid); 
			}
			else
			{
				_orderedUnit = orderedUnit;
			}

			if (orderedComponent == null)
			{
                //CR 3554
                DataTable dtOrderedComponent = new DataTable();
			    if (_orderedUnit != null)
			    {
			        dtOrderedComponent = BOL.OrderedComponent.GetOrderedComponent(_orderedUnit.OrderedComponentGuid);
			        if (dtOrderedComponent != null)
			        {
			            _orderedComponent = dtOrderedComponent.Rows.Count > 0
			                ? new OrderedComponent(dtOrderedComponent.Rows[0])
			                : new OrderedComponent(dtOrderedComponent.NewRow());
			        }
			    }
			}
			else
			{
				_orderedComponent = orderedComponent;
			}
			// CR2899 end


			_division = new Division(Common.LogonUser.LogonUserDivisionCode);
			_isFullService = _division.ServiceTypeIndicator;
			
			_exceptionReportEntries = new ArrayList();

			
			
			_message = new StringBuilder();
			_formMessage = new StringBuilder();

			if (rulesAlreadyValidated)
			{
				_emergencyIssue = IsEmergencyIssue();

				if (_emergencyIssue)
				{
					// CR2899: start
                    //CR 3554
                    if (_orderedComponent != null && _orderedUnit != null)
                    {
                        Specimen spec = _orderedComponent.GetSpecimen();

                        if (!bloodUnit.EligibleForEmergencyIssue(
                            _orderedUnit.CrossmatchRequiredIndicator,
                            _patient.ABORhSource == Common.ABORhSource.ConversionDiscrepancy ||
                            _patientABORhSourceCache == Common.ABORhSource.SpecimenTestDiscrepancy ||
                            _orderedComponent.SpecimenIdentified &&
                            spec.RepeatABORhTestsMatch == Common.TypingTestResults.Discrepancy, // CR3114
                            _orderedComponent.SpecimenIdentified && spec.ABORhTested)) // CR3114
                        // CR2899: end
                        {
                            throw new BOL.BusinessObjectException(Common.StrRes.SysErrMsg.UC043.CannotEmergencyIssue().ResString);
                        }
                    }
				}
				_overrideRequired = false;
				_unitSelectable = true;
			}
			else
			{
				// CR2899 start: pass in cache objects
				Validate(antigenNegsReqsCache, componentsCache);
				// CR2899 end
			}
		}

		#endregion


		#region properties

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/8/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7418"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="7419"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Inidicates whether an audible alert is needed.
		/// </summary>
		public bool Beep
		{
			get
			{
				return _beep;
			}
		}

		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/28/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4604"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>The blood unit returned by this property will match 
		///		the unit passed into the constructor</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="1751"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Unit that tag/ for pertains to
		/// </summary>
		public BOL.BloodUnit BloodUnit
		{
			get
			{
				return _bloodUnit;
			}
		}

		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/28/2004</CreationDate>
		///<TestCases>
		///
		/// <Case type="0" testid ="4605"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>The patient returned by this property will match 
		///		the patient passed into the constructor</ExpectedOutput>
		///	</Case>
		///
		/// <Case type="1" testid ="1752"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Patient that tag/ form pertains to
		/// </summary>
		public BOL.Patient Patient
		{
			get
			{
				return _patient;
			}
		}

		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>6/1/2004</CreationDate>
		///
		///<TestCases>
		///	
		///<Case type="0" testid ="7420"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="7421"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// All messages associated with this tag
		/// </summary>
		public String Message
		{
			get
			{
				return _message.ToString();
			}
		}

		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>6/1/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4618"> 
		///		<ExpectedInput>Valid (non empty) string</ExpectedInput>
		///		<ExpectedOutput>Property set to string with no broken rules</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4619"> 
		///		<ExpectedInput>Empty string</ExpectedInput>
		///		<ExpectedOutput>Broken rule count incremented by 1</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// OverrideComment
		/// </summary>
		public string OverrideComment
		{
			get
			{
				return _overrideComment;
			}
			set
			{
				_overrideComment = value;
				RuleBroken(MethodBase.GetCurrentMethod().Name, value.Trim().Length==0);

				for (int i = 0; i < _exceptionReportEntries.Count; i++)
				{
					((BOL.ExceptionReport) _exceptionReportEntries[i]).ExceptionComment = _overrideComment;
					((BOL.ExceptionReport) _exceptionReportEntries[i]).GenerateExceptionCautionTagBTRF(_patient, _orderedComponent.SpecimenIdentified ? _orderedComponent.GetSpecimen() : null, _bloodUnit);  // CR3114
				}
			}
		}

		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/8/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7422"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>bool</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="7423"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Indicates whether or not a tag/ form can be printed.
		/// </summary>
		public bool PrintOK
		{
			get
			{
				return _unitSelectable;
			}
		}

		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/8/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7424"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>bool</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="7425"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Indicates whether or not an override comment is required.
		/// </summary>
		public bool OverrideRequired
		{
			get
			{
				return _overrideRequired;
			}
		}

		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/8/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7426"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>bool</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="7427"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Emergency Issue
		/// </summary>
		public bool EmergencyIssue
		{
			get
			{
				return _emergencyIssue;
			}
		}

		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/8/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7430"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>ArrayList</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="7431"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// ExceptionReportEntries
		/// </summary>
		public ArrayList ExceptionReportEntries
		{
			get
			{
				return _exceptionReportEntries;
			}
		}
		
		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/8/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7428"> 
		///		<ExpectedInput>bool</ExpectedInput>
		///		<ExpectedOutput>bool</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="7429"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Set to true after one entry generated to prevent duplicate entries.
		/// </summary>
		public bool ExceptionReportGenerated
		{
			get
			{
				return _exceptionReportGenerated;
			}
			set
			{
				_exceptionReportGenerated = value;
			}
		}

		#endregion


		#region methods

		#region static methods

		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/28/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4606"> 
		///		<ExpectedInput>Guid for patient that has units</ExpectedInput>
		///		<ExpectedOutput>true</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4607"> 
		///		<ExpectedInput>New Guid</ExpectedInput>
		///		<ExpectedOutput>false</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Returns a boolean indicating whether or not the patient has units
		/// available for issue (or printing in this UC)
		/// </summary>
		/// <param name="patientGuid"></param>
		/// <returns></returns>
		public static bool VerifyPatientHasUnits(Guid patientGuid)
		{
			if (patientGuid == Guid.Empty)
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat("ZPL").ResString);
			}

			DataTable dtOrderedUnits = BOL.OrderedUnit.GetOrderedUnitsForPatient(patientGuid);
			return dtOrderedUnits.Rows.Count > 0;
		}


		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/28/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4608"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>DataTable with at least 1 row</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4609"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// GetUnitsForDisplay
		/// BR_43.01
		/// </summary>
		/// <returns></returns>
		public static DataTable GetUnitsForDisplay()
		{
			return BOL.OrderedUnit.GetOrderedUnitsForUnit();
		}


		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/28/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4610"> 
		///		<ExpectedInput>A valid patient Guid that has units associated with it</ExpectedInput>
		///		<ExpectedOutput>At least one unit returned (ArrayList with length greater 
		///		than 0.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4611"> 
		///		<ExpectedInput>An Guid not associated with any patient.</ExpectedInput>
		///		<ExpectedOutput>No units returned (ArrayList with length of 0).</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Returns units associated with the patient.
		/// </summary>
		/// <returns></returns>
		public static ArrayList GetUnitsForDisplay(Guid patientGuid)
		{
			DataTable dtOrderedUnits = BOL.OrderedUnit.GetOrderedUnitsForPatient(patientGuid);
			ArrayList displayUnits = new ArrayList(dtOrderedUnits.Rows.Count);
			foreach (DataRow dtRow in dtOrderedUnits.Rows)
			{
				// Oooooh; can't send DataTables to the UI layer!
				Hashtable ht = new Hashtable();
				ht.Add(TABLE.BloodUnit.BloodUnitGuid, 
					dtRow[TABLE.BloodUnit.BloodUnitGuid]);
				ht.Add(TABLE.BloodUnit.EyeReadableUnitId, 
					dtRow[TABLE.BloodUnit.EyeReadableUnitId].ToString() );
				ht.Add(TABLE.BloodProduct.ProductShortName,
					dtRow[TABLE.BloodProduct.ProductShortName].ToString() );
				ht.Add(ARTIFICIAL.ABORh,
					dtRow[ARTIFICIAL.ABORh].ToString() );
				ht.Add(TABLE.BloodUnitMedia.UnitExpirationDate, 
					dtRow[TABLE.BloodUnitMedia.UnitExpirationDate].ToString() );
				displayUnits.Add(ht);
			}
			return displayUnits;
		}


		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/28/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4612"> 
		///		<ExpectedInput>A bloodUnitGuid associated with at least one patient.</ExpectedInput>
		///		<ExpectedOutput>At least one patient returned (ArrayList with a length 
		///		greater than 0).</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4613"> 
		///		<ExpectedInput>A bloodUnitGuid that has no patient associations.</ExpectedInput>
		///		<ExpectedOutput>No patients returned (ArrayList with length of 0).</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Returns a list of patients that have units associated with the unit.
		/// </summary>
		/// <param name="bloodUnitGuid"></param>
		/// <returns></returns>
		public static ArrayList GetPatients(Guid bloodUnitGuid)
		{
			DataTable dtOrderedUnits = BOL.OrderedUnit.GetOrderedUnit(bloodUnitGuid, Common.LogonUser.LogonUserDivisionCode);
			ArrayList patients = new ArrayList(dtOrderedUnits.Rows.Count);
			foreach (DataRow dtRow in dtOrderedUnits.Rows)
			{
				patients.Add(new BOL.Patient( (Guid) dtRow[TABLE.Patient.PatientGuid] ));
			}
			return patients;
		}


		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/28/2004</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="4620"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4621"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Prints blank caution tags
		/// BR_43.11, BR_43.15
		/// </summary>
		/// <returns></returns>
		public static void PrintBlankCautionTag()
		{
			new CautionTag().Print(CautionTag.BLANK_TAG, true);
		}


		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>6/1/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4622"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4623"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Prints blank blood transfusion record forms
		/// </summary>
		/// <param name="number">specifies number of tags to print</param>
		/// <returns></returns>
		public static int PrintBlankBtrfs(int number)
		{
			int count = 0;
			while(count < number)
			{
				if (!new CautionTag().Print(Common.Print.BTRF)) return count;
				Thread.Sleep(1500);
				count++;
			}
			return count;
		}

		#endregion


		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/28/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4616"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>DataSet containing 1 table where all values are empty strings</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4617"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Returns an empty DataSet for printing backup (blank) forms
		/// </summary>
		/// <returns></returns>
		public static DataSet GetEmptyBtrfDataSet()
		{
			DataSet dsBtrfDataSet = new DataSet();
			dsBtrfDataSet.Tables.Add( new CautionTag().GetBtrfDataTable());
			return dsBtrfDataSet;
		}

		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>6/1/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4624"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4625"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Prints the caution tag.
		///
		/// </summary>
		/// <returns></returns>
		public bool Print(Common.Print print)
		{
			if(!_unitSelectable)
			{
				throw new BOL.BusinessObjectException("Caution Tag cannot be printed.  " + 
					"Please check the Message propery for an explanation.");
			}
			else if ( _overrideRequired == true &&  (_overrideComment== null || _overrideComment.Trim() == string.Empty) )
			{
				throw new BOL.BusinessObjectException("Caution Tag can be printed, but .  " + 
					"requires an override comment.  Please check the Message propery for " +
					"an explanation.");
			}
			else
			{
				// BR_43.13
				if (!Common.LogonUser.PrintedTags.ContainsKey(_bloodUnit.BloodUnitGuid))
				{	
					Common.LogonUser.PrintedTags.Add(_bloodUnit.BloodUnitGuid, _patient.PatientGuid);
				}
				bool success = false;
				switch (print)
				{
					case Common.Print.CautionTag:
						success =  Print( FormatZPL(), false );
						break;
				}
				return success;
			}
		}

		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>6/1/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4626"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>A DataSet for the printing of the BTRF</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="9160"> 
		///		<ExpectedInput>Patient with antibodies coming from VBECS and from conversion</ExpectedInput>
		///		<ExpectedOutput>string containing both types of antibodies</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4627"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// BR_44.01, BR_44.02, BR_44.03, and BR_44.05
		/// Returns DataSet used for report
		/// </summary>
		/// <returns></returns>
		public DataSet GetBtrfDataSet()
		{
			StringBuilder remarks = new StringBuilder();

			DataSet dsBtrfData = new DataSet("BTRF");
			DataTable dtBtrfData = this.GetBtrfDataTable();

			DataRow dtRow = dtBtrfData.NewRow();
			
			if (IsEmergencyIssue()) 
			{
				// BR_44.05
				remarks.Append(Common.StrRes.InfoMsg.UC044.CompatibilityTestingNotFinished().ResString);
				
				dtRow[ARTIFICIAL.EmergencyIssue] = true;
				
				if (!_orderedComponent.SpecimenRequiredIndicator ||
					(_orderedComponent.SpecimenIdentified && _orderedComponent.GetSpecimen().ABORhTested)) // CR3114
				{
					dtRow[ARTIFICIAL.ABORh] = _patient.AboRh.AboRHText;
				}
				else
				{
					dtRow[ARTIFICIAL.ABORh] = "Unknown";
					remarks.Append("*Specimen testing is incomplete at time of issue.\n");
				}
			}
			else
			{
				dtRow[ARTIFICIAL.EmergencyIssue] = false;
				dtRow[ARTIFICIAL.ABORh] = _patient.AboRh.AboRHText;
			}

			// Add text related to 43.18 and 43.19 to the remarks section
			remarks.Append(FormatRemarks());
		
			
			Address address = new Address( _user.CurrentDivisionRole.Division.FacilityGuid );

			// Put all this into DivisionName to eliminate spacing issues on the form			
			string division =  _user.CurrentDivisionRole.Division.DivisionName + ", " + 
				address.City+ ", " + address.StateCode;

			// BR_44.02 - Include the testing facility information
			if (!_isFullService)
			{
				division = division + "\nTesting facility: " + _bloodUnit.IncomingShipment.LocalSupplier.FacilityName + ", " + _bloodUnit.IncomingShipment.LocalSupplier.Address.City + 
					", " + _bloodUnit.IncomingShipment.LocalSupplier.Address.StateCode;
			}

			dtRow[TABLE.VamcDivision.DivisionName] = division;

			// CR_2419: DisplayName was put here so middle initial shows up 
			dtRow[TABLE.Patient.PatientLastName]		= _patient.DisplayName;
			// end CR_2419
			dtRow[TABLE.Patient.VistaPatientId]		= _patient.DisplayPatientID;
			dtRow[TABLE.Patient.PatientDob]			= _patient.DateOfBirth;
				
			dtRow[TABLE.BloodUnit.EyeReadableUnitId]	= GetFormattedUnit(_bloodUnit);

            dtRow[TABLE.BloodProduct.ProductCode] = _bloodUnit.FullProductCode; // CR 3565

			dtRow[TABLE.Compatibility.UnitAboRh] = _bloodUnit.BloodUnitMedia.AboRh.AboRHText;

            dtRow[TABLE.BloodUnit.DonationTypeId] = _bloodUnit.DonationType.DonationTypeText;  // DR 3565

			// CR2899 start: changed from unit guid to string
			DataRow dr = DAL.OrderedUnit.GetOrderedUnitByPatientAndUnit(_bloodUnit.BloodUnitGuid.ToString(), _patient.PatientGuid).Rows[0];
			// CR2899 end
			if (_orderedUnit.CrossmatchRequiredIndicator)
			{
				string crossmatchText = string.Empty;
				string tech = string.Empty;

				// BR_43.04
				if (_orderedComponent.SpecimenIdentified == false || !_orderedComponent.GetSpecimen().TASTested) // CR3114
				{
					crossmatchText = Common.StrRes.OtherMsg.UC043.NoCrossmatchAtIssue().ResString;
				}
				else if (_orderedUnit.CurrentCrossmatchDate > DateTime.MinValue) 
				{
					if (_orderedUnit.CurrentCrossmatchTechId.Trim() != string.Empty) tech = new BOL.VbecsUser(dr[TABLE.OrderedUnit.CrossmatchTechId].ToString()).UserName;
					
					crossmatchText = Common.Utility.GetCautionTagCrossmatchText(Common.Utility.GetCrossmatchResultEnumFromString(_orderedUnit.CurrentCrossmatchResultCode)) + 
						", " + Common.VBECSDateTime.FormatDateTimeString(_orderedUnit.CurrentCrossmatchDate) + ", " + tech;
				}
				else
				{
					crossmatchText = "TESTING INCOMPLETE";
				}
				dtRow[TABLE.CrossmatchResult.CrossmatchResultText] = crossmatchText;
			}
			else
			{
				dtRow[TABLE.CrossmatchResult.CrossmatchResultText] = "Not required";
			}
			dtRow[TABLE.OrderedUnit.SelectedDate]	= Common.VBECSDateTime.FormatDateTimeString(_orderedUnit.SelectedDate);
			// Get the initials
			dtRow[TABLE.OrderedUnit.SelectedTechId]	= _orderedUnit.SelectedTechId;

			if (_orderedUnit.SelectedTechId != string.Empty) dtRow[TABLE.VbecsUser.UserInitials] = new BOL.VbecsUser(_orderedUnit.SelectedTechId).UserInitials;
			
			dtRow[ARTIFICIAL.PatientTRANString] = this.FormatTrString();
			dtRow[TABLE.BloodProduct.ProductName] = _bloodUnit.ProductName;
			dtRow[TABLE.BloodUnitMedia.UnitExpirationDate] = Common.VBECSDateTime.FormatDateTimeString(_bloodUnit.BloodUnitMedia.UnitExpirationDate);
			dtRow[ARTIFICIAL.PooledUnitsCount] = _bloodUnit.PooledUnitsCount;

			string uid = string.Empty;
			if (!dr.IsNull(TABLE.PatientSpecimen.SpecimenUid)) 
				dtRow[TABLE.PatientSpecimen.SpecimenUid] = dr.IsNull(TABLE.PatientSpecimen.SpecimenUid) ? "No specimen" : dr[TABLE.PatientSpecimen.SpecimenUid].ToString();

            dtRow[TABLE.PatientSpecimen.SpecimenUid] = dr[TABLE.PatientSpecimen.SpecimenUid].ToString() + ", " + Common.VBECSDateTime.FormatDateTimeString(dr[TABLE.PatientSpecimen.SpecimenExpirationDate]); // CR 3565

			dtRow[TABLE.PatientTreatment.PatientLocation] = dr.IsNull(TABLE.PatientTreatment.PatientLocation) ? string.Empty : dr[TABLE.PatientTreatment.PatientLocation].ToString();
			dtRow[ARTIFICIAL.BTRFRemarks] = remarks.ToString();

			dtBtrfData.Rows.Add(dtRow);
			dsBtrfData.Tables.Add(dtBtrfData);

			return dsBtrfData;	
		}

		/// <summary>
		/// Returns an empty DataTable representing the information contained in the BTRF
		/// </summary>
		/// <returns></returns>
		private DataTable GetBtrfDataTable()
		{
			DataTable dtBtrfData = new DataTable("BTRF");

			dtBtrfData.Columns.Add(ARTIFICIAL.EmergencyIssue, typeof(byte));
			dtBtrfData.Columns.Add(TABLE.VamcDivision.DivisionName, typeof(string));
			dtBtrfData.Columns.Add(TABLE.Patient.PatientLastName, typeof(string));
			dtBtrfData.Columns.Add(TABLE.Patient.VistaPatientId, typeof(string));
			dtBtrfData.Columns.Add(TABLE.Patient.PatientDob, typeof(string));
			dtBtrfData.Columns.Add(ARTIFICIAL.ABORh, typeof(string));
			dtBtrfData.Columns.Add(TABLE.BloodUnit.EyeReadableUnitId, typeof(string));
			dtBtrfData.Columns.Add(TABLE.Compatibility.UnitAboRh, typeof(string));

            dtBtrfData.Columns.Add(TABLE.BloodUnit.DonationTypeId, typeof(string)); // CR 3565

			dtBtrfData.Columns.Add(TABLE.OrderedUnit.CrossmatchResultCode, typeof(string)); // Convert this to text
			dtBtrfData.Columns.Add(TABLE.OrderedUnit.SelectedDate, typeof(string));
			dtBtrfData.Columns.Add(TABLE.OrderedUnit.SelectedTechId, typeof(string));
			dtBtrfData.Columns.Add(TABLE.VbecsUser.UserInitials, typeof(string));
			dtBtrfData.Columns.Add(TABLE.CrossmatchResult.CrossmatchResultText, typeof(string));
			
			dtBtrfData.Columns.Add(ARTIFICIAL.PatientTRANString, typeof(string));
			
			dtBtrfData.Columns.Add(TABLE.BloodProduct.ProductName, typeof(string));

            dtBtrfData.Columns.Add(TABLE.BloodProduct.ProductCode, typeof(string)); // CR 3565

			dtBtrfData.Columns.Add(TABLE.BloodUnitMedia.UnitExpirationDate, typeof(string));
			dtBtrfData.Columns.Add(ARTIFICIAL.PooledUnitsCount, typeof(string));

			dtBtrfData.Columns.Add(TABLE.PatientSpecimen.SpecimenUid, typeof(string));

            dtBtrfData.Columns.Add(TABLE.PatientSpecimen.SpecimenExpirationDate, typeof(string));  // CR 3565

			dtBtrfData.Columns.Add(TABLE.PatientTreatment.PatientLocation, typeof(string));
			dtBtrfData.Columns.Add(ARTIFICIAL.BTRFRemarks, typeof(string));

            
		
			return dtBtrfData;
		}

		/// <summary>
		/// Formats the string for transfusion requirements
		/// </summary>
		/// <returns></returns>
		private string FormatTrString()
		{
			ArrayList duplicateComments = new ArrayList();
			bool systemGeneratedANRequirementsExist = false;

			StringBuilder trString = new StringBuilder();
			// Component requirements
			DataRow [] dtRowCrs =  _patient.TransfusionRequirements.Select(TABLE.PatientTransfusionRequirement.RecordStatusCode + " = 'A' AND " +  TABLE.PatientTransfusionRequirement.TransfusionRequirementCategoryCode + " = 'CR'");
			// Antigen negative requirements
			DataRow [] dtRowANs =  _patient.TransfusionRequirements.Select(TABLE.PatientTransfusionRequirement.RecordStatusCode + " = 'A' AND " +  TABLE.PatientTransfusionRequirement.TransfusionRequirementCategoryCode + " = 'AN'");
			// Antibodies
			//CR 2907 Added antibodies that have a status of C (from conversion)
			DataRow [] dtRowABs =  _patient.TransfusionRequirements.Select("(" + TABLE.PatientTransfusionRequirement.RecordStatusCode + " = 'A' OR " + TABLE.PatientTransfusionRequirement.RecordStatusCode + " = 'C') AND " +  TABLE.PatientTransfusionRequirement.TransfusionRequirementCategoryCode + " = 'AS'");
			
			// Component requirements
			if (dtRowCrs.Length > 0)
			{
				trString.Append("COMPONENT REQS: ");
				for (int i=0; i< dtRowCrs.Length; i++)
				{
					trString.Append(dtRowCrs[i][TABLE.PatientTransfusionRequirement.TransfusionRequirementText].ToString());
					if (i < dtRowCrs.Length-1) trString.Append(", ");
				}
			}

			// Antigen negative requirements
			if (dtRowANs.Length > 0)
			{
				if (trString.Length != 0) trString.Append("  "); 
				trString.Append("ANTIGEN NEGATIVE REQS: ");

				for (int i=0; i<dtRowANs.Length; i++)
				{
					trString.Append(dtRowANs[i][TABLE.PatientTransfusionRequirement.TransfusionRequirementText].ToString());
					if (i < dtRowANs.Length-1) trString.Append(", ");
					duplicateComments.Add(dtRowANs[i][TABLE.PatientTransfusionRequirement.TransfusionRequirementText].ToString());
				}
			}

			// System generated antigen negative requirements
			DataTable dt = BOL.AntibodyType.GetAntibodyTypeLookupList();
			DataTable dtAntibodyType		= BOL.AntibodyType.GetAntibodyType();
			foreach(DataRow dr in dtRowABs) 
			{
				int antibodyTypeID = (int) dr[TABLE.AntibodyType.AntibodyTypeId];
				DataRow [] dtAntibodyTypes = dtAntibodyType.Select(TABLE.AntibodyType.AntibodyTypeId + " = " + antibodyTypeID);
				// CR2483
				if (dtAntibodyTypes.Length == 0) continue;
				// CR2483 end
				string antibodyTypeName = (string) dtAntibodyTypes[0][TABLE.AntibodyType.AntibodyTypeName];

				string filter = TABLE.AntibodyType.AntibodyTypeId + " = " + antibodyTypeID + " AND " + TABLE.AntibodyParameter.HigherLevelOverrideIndicator + " = 1";
				DataRow[] dtOverride = dt.Select(filter);

				if (dtOverride.Length > 0)
				{
					DataRow [] dtAntNegs = dtAntibodyType.Select(TABLE.AntibodyType.AntibodyTypeName + " = '" + dtOverride[0][TABLE.AntibodyType.AntibodyTypeName] + "'");

					foreach (DataRow dtRow in dtAntNegs)
					{
						if ( !dtRow.IsNull(TABLE.AntigenType.AntigenTypeName) 
							&& !dtRow[TABLE.AntigenType.AntigenTypeName].ToString().ToUpper().Equals("none")
							&& dtRow[TABLE.AntibodyType.AntibodyTypeName].ToString() == antibodyTypeName)
						{
							if (duplicateComments.Contains( dtRow[TABLE.AntigenType.AntigenTypeName].ToString().Trim() )) continue;
							
							// formatting
							if (dtRowANs.Length == 0 && systemGeneratedANRequirementsExist == false )
							{
								trString.Append("  ANTIGEN NEGATIVE REQS: ");
							}
							else 
							{
								trString.Append(", ");
							}
							string tr = dtRow[TABLE.AntigenType.AntigenTypeName].ToString();
							trString.Append(tr);
							systemGeneratedANRequirementsExist = true;
							duplicateComments.Add( dtRow[TABLE.AntigenType.AntigenTypeName].ToString().Trim() );
						}
					}
				}
			}
			return trString.ToString();
		}

		/// <summary>
		/// FormatRemarks
		/// </summary>
		/// <returns></returns>
		private string FormatRemarks()
		{
			StringBuilder remarks = new StringBuilder();

			DataRow [] dtRowCrs =  _patient.TransfusionRequirements.Select(TABLE.PatientTransfusionRequirement.RecordStatusCode + " = 'A' AND " +  TABLE.PatientTransfusionRequirement.TransfusionRequirementCategoryCode + " = 'CR'");

			foreach (DataRow dtRow in dtRowCrs)
			{
				BOL.PatientTransfusionRequirement pTR = new BOL.PatientTransfusionRequirement(dtRow);
				if (pTR.ComponentRequirementId == int.MinValue) continue;
				
				if (!_bloodUnit.IsComponentRequirementSatisfied( (Common.ComponentRequirement) pTR.ComponentRequirementId))
				{
					remarks.Append("This unit is NOT " + GetRequirementText((Common.ComponentRequirement)pTR.ComponentRequirementId) + ".  ");
				}
			}

			if (!BOL.BloodUnit.IsAutologous(_bloodUnit.DonationType.DonationTypeCode) && 
				(_bloodUnit.BloodComponentClassId == (int) Common.ComponentClass.RBC ||
				_bloodUnit.BloodComponentClassId == (int) Common.ComponentClass.WB))
			{
				ArrayList anReqs = _patient.ValidateAntibodyAntigenRequirementsMet(_bloodUnit);
				if (anReqs.Count > 0)
				{
					remarks.Append("\n");
					foreach (BOL.AntigenNegativeMessage aNM in anReqs)
					{
						remarks.Append(Common.StrRes.InfoMsg.UC043.NotAntigenNegative(aNM.AntigenName).ResString);
					}
				}
			}

			if (remarks.Length>0) remarks.Insert(0, "*Caution: ");
			return remarks.ToString();
		}

		/// <summary>
		/// GetRequirementText
		/// </summary>
		/// <param name="req"></param>
		/// <returns></returns>
		private string GetRequirementText(Common.ComponentRequirement req)
		{
			string reqString = string.Empty;

			switch (req)
			{
				case Common.ComponentRequirement.CMVNEG:
					reqString = "CMV NEGATIVE";
					break;
				case Common.ComponentRequirement.IRRADIATE:
					reqString = "IRRADIATED";
					break;
				case Common.ComponentRequirement.LEUKORED:
					reqString = "LEUKOREDUCED";
					break;
				case Common.ComponentRequirement.SCNEG:
					reqString = "SICKLE CELL NEGATIVE";
					break;
				case Common.ComponentRequirement.WASHEDPLT:
					reqString = "WASHED (Platelet)";
					break;
				case Common.ComponentRequirement.WASHEDRBC:
					reqString = "WASHED (RBC)";
					break;
			}
			return reqString;
		}

		/// <summary>
		/// Determines if the tag is printable and if so, what overrides are required.
		/// BR_43.02, BR_43.07, BR_43.13, BR_43.14, BR_43.16, BR_43.17, BR_43.18, BR_43.19
		/// </summary>
		/// CR2899: now, we pass in cached version of OrderedComponent
		private void Validate(Hashtable antigenNegsReqsCache, Hashtable componentsCache)
		{
			_exceptionReportEntries.Clear();
				
			Common.ComponentClass componentClass = (Common.ComponentClass) _bloodUnit.BloodComponentClassID;

			#region Unit unselectable

			// CR2908 start
			// Invalid status
			if (_bloodUnit.BloodUnitStatus.UnitStatusCode != Common.UnitStatusCode.Assigned &&
				_bloodUnit.BloodUnitStatus.UnitStatusCode != Common.UnitStatusCode.CrossmatchedAssigned)
			{
				_message.Append(Common.StrRes.ValidMsg.UC003.BadStatus(Common.Utility.GetUnitStatusTextFromEnum(_bloodUnit.BloodUnitStatus.UnitStatusCode)).ResString);
				_unitSelectable = false;
				return;
			}
			// CR2908 end


			// BR_73.35 - inconclusive XM
			if (_orderedUnit.CrossmatchRequiredIndicator)
			{
				// BR_43.21
				Common.CrossmatchResult crossmatchResult = Common.Utility.GetCrossmatchResultEnumFromString( _orderedUnit.CurrentCrossmatchResultCode );
				if (crossmatchResult == Common.CrossmatchResult.Inconclusive)
				{
					_message.Append(Common.StrRes.InfoMsg.UC043.InconclusiveCrossmatch().ResString);
					_unitSelectable = false;
					return;
				}
			}


			// BR_43.13 - Already selected for a different user in the session
			if (Common.LogonUser.PrintedTags.ContainsKey(_bloodUnit.BloodUnitGuid))
			{
				if ( (Guid) Common.LogonUser.PrintedTags[_bloodUnit.BloodUnitGuid] != _patient.PatientGuid)
				{
					_message.Append(Common.StrRes.InfoMsg.UC043.TagsAlreadyPrinted().ResString);
					_unitSelectable = false;
					return;
				}
			}

			
			// BR_43.02, BR_43.07 - Frozen unit
			if (_bloodUnit.IsFrozen)
			{
				_message.Append(Common.StrRes.InfoMsg.UC043.UnitFrozen().ResString);
				_unitSelectable = false;
				return;
			}
			

			// Incompatible crossmatch result
			Common.CrossmatchResult xmResult = Common.Utility.GetCrossmatchResultEnumFromString( _orderedUnit.CurrentCrossmatchResultCode );
			if (_orderedUnit.CrossmatchRequiredIndicator == true 
				&& (xmResult == Common.CrossmatchResult.CompatibleDontTransfuse ||
				xmResult == Common.CrossmatchResult.IncompatibleUnsafeToTransfuse ))
			{
				_message.Append(Common.StrRes.InfoMsg.UC043.IncompatibleCrossmatchResults().ResString);
				_unitSelectable = false;
				return;
			}

			
			// ABO/Rh confirmations do not match
            //CR3558 - confirmation testing is now required on more than RBC/WB, so checking the PT1.06 indicator
            if (_bloodUnit.ProductType.AboRhConfirmationRequired &&
				_isFullService &&
				!_bloodUnit.BloodUnitMedia.DoAboRHConfirmationsMatch(Common.LogonUser.LogonUserDivisionCode) )
			{
				_message.Append(Common.StrRes.InfoMsg.UC043.UnitABORhDiscrepancy().ResString);
				_unitSelectable = false;
				return;
			}


			// Unit quarantined
			if (_bloodUnit.BloodUnitStatus.QuarantineIndicator)
			{
				_message.Append(Common.StrRes.InfoMsg.UC043.UnitQuarantined().ResString);
				_unitSelectable = false;
				return;
			}
	

			// BR_43.20
			_emergencyIssue = IsEmergencyIssue();
			// CR2899: start
			
			if (_emergencyIssue && !_bloodUnit.EligibleForEmergencyIssue(
				_orderedUnit.CrossmatchRequiredIndicator, 
				_patient.ABORhSource == Common.ABORhSource.ConversionDiscrepancy ||
				_patientABORhSourceCache == Common.ABORhSource.SpecimenTestDiscrepancy || 
				_orderedComponent.SpecimenIdentified && _orderedComponent.GetSpecimen().RepeatABORhTestsMatch == Common.TypingTestResults.Discrepancy,  // CR3114
				_orderedComponent.SpecimenIdentified && _orderedComponent.GetSpecimen().ABORhTested))  // CR3114
				
			{
				_message.Append(Common.StrRes.InfoMsg.UC043.EmergencyIssueInvalid().ResString);
				_unitSelectable = false;
				return;
			}
			
			// CR2899: end
			
			#endregion


			#region Unit selectable with overrides

			// Unit Antigen Negative (or clinically significant antibody) requirements, level 3 
			// 43.14, 43.18
			if (Common.ComponentClass.WB == (Common.ComponentClass) _bloodUnit.BloodComponentClassID ||
				Common.ComponentClass.RBC == (Common.ComponentClass) _bloodUnit.BloodComponentClassID )
			{
				// CR2899 start: get results from cache object if available
				ArrayList antigenNegReqs = null;
				if (antigenNegsReqsCache != null)
				{
					antigenNegReqs = (ArrayList) antigenNegsReqsCache[_bloodUnit.BloodUnitGuid];
				}
				else
				{
					antigenNegReqs = _patient.ValidateAntibodyAntigenRequirementsMet(_bloodUnit);
				}
				// CR2899 end
				
				foreach (BOL.AntigenNegativeMessage aNM in antigenNegReqs)
				{
					if (!aNM.Tested  || (aNM.Tested && aNM.TestResult == Common.TestResult.P) )
					{			
						_beep = true;

						if (HasAuthority(Common.FunctionCode.IssueAntigenPosOrUntypedRedbloodcells))
						{
							// Antigen
							if (aNM.AntibodyName.Trim() == string.Empty)
							{
								_message.Append(Common.StrRes.InfoMsg.UC043.AntigenProblemCautionTag(aNM.AntigenName, _bloodUnit.EyeReadableUnitId).ResString + "\n");
							}
							// Antibody
							else
							{
								_message.Append(Common.StrRes.InfoMsg.UC043.AntibodyProblemCautionTag(aNM.AntibodyName, _bloodUnit.EyeReadableUnitId, aNM.AntigenName).ResString + "\n");
							}
							_overrideRequired = true;
							_exceptionReportEntries.Add(new BOL.ExceptionReport(Common.ExceptionType.CautionTag_BTRF_antigen_neg_reqs_not_met, string.Empty, Common.LogonUser.LogonUserName));
						}
						else
						{
							_message.Remove(0, _message.Length);
							_unitSelectable = false;
							// ToDo: DR 1724 has been submitted to address this bad message... 
							if (aNM.AntibodyName.Trim() == string.Empty)
							{
								_message.Append(Common.StrRes.InfoMsg.UC043.AntigenProblemNoOverride(aNM.AntigenName, _bloodUnit.EyeReadableUnitId).ResString);
							}
							else
							{
								_message.Append(Common.StrRes.InfoMsg.UC043.AntibodyProblemNoOverride(aNM.AntibodyName, _bloodUnit.EyeReadableUnitId, aNM.AntigenName).ResString);
							}
							return;
						}
					}
				}
			}

			

			// Unit Transfusion requirement, level 3
			// BR_43.19, TT_3.04
			ArrayList cRs = null;
			// CR2899 start: get results from cache object if available
			if (componentsCache != null)
			{
				cRs = (ArrayList) componentsCache[_bloodUnit.BloodUnitGuid];
			}
			else
			{
				cRs = _patient.ValidateComponentRequirements(_bloodUnit.BloodUnitGuid);
			}
			// CR2899 end
			if (cRs != null && cRs.Count > 0)
			{
				_beep = true;

				StringBuilder cRString = new StringBuilder();
				foreach (string cR in cRs)
				{
					if (cRString.Length > 0) cRString.Append(", ");
					cRString.Append(cR);
				}

				if (HasAuthority(Common.FunctionCode.IssueAntigenPosOrUntypedRedbloodcells))
				{
					_message.Append(Common.StrRes.InfoMsg.UC043.TRNotSatisfied(cRString.ToString()).ResString + "\n" );
					_overrideRequired = true;
					_exceptionReportEntries.Add(new BOL.ExceptionReport(Common.ExceptionType.CautionTag_BTRF_component_reqs_not_met, string.Empty, Common.LogonUser.LogonUserName));
				}
				else
				{
					_message.Remove(0, _message.Length);
					_unitSelectable = false;
					_message.Append(Common.StrRes.InfoMsg.UC043.TRNotSatisfiedNoOverride(cRString.ToString()).ResString);
					return;
				}
			}



			// Specimen expiration date, level 2
			// BR_43.02, BR_43.16
			if (_orderedComponent.SpecimenIdentified && _orderedComponent.GetSpecimen().SpecimenExpirationDate < BOL.VBECSDateTime.GetDivisionCurrentDateTime())  // CR3114
			{
				_beep = true;

				if (HasAuthority(Common.FunctionCode.IssueExpiredBloodProducts))
				{
					_message.Append(Common.StrRes.InfoMsg.UC043.SpecimenExpired().ResString + "\n" );
					_overrideRequired = true;
					_exceptionReportEntries.Add(new BOL.ExceptionReport(Common.ExceptionType.CautionTag_BTRF_expired_specimen, string.Empty, Common.LogonUser.LogonUserName));
				}
				else
				{
					_message.Remove(0, _message.Length);
					_unitSelectable = false;
					_message.Append(Common.StrRes.InfoMsg.UC043.SpecimenExpiredNoOverride().ResString);
					return;
				}
			}

			

			// Unit expiration date, level 2
			// BR_43.17
			if (_bloodUnit.BloodUnitMedia.UnitExpirationDate < BOL.VBECSDateTime.GetDivisionCurrentDateTime())
			{
				_beep = true;
				
				if (HasAuthority(Common.FunctionCode.IssueExpiredBloodProducts))
				{
					_message.Append(Common.StrRes.InfoMsg.UC043.UnitExpired().ResString + "\n" );
					_overrideRequired = true;
					_exceptionReportEntries.Add(new BOL.ExceptionReport(Common.ExceptionType.CautionTag_BTRF_expired_unit, string.Empty, Common.LogonUser.LogonUserName));
				}
				else
				{
					_message.Remove(0, _message.Length);
					_unitSelectable = false;
					_message.Append(Common.StrRes.InfoMsg.UC043.UnitExpiredNoOverride().ResString);
					return;
				}
			}


			#endregion


			#region Unit selectable with warnings

			if (((Common.ComponentClass) _bloodUnit.BloodComponentClassId == Common.ComponentClass.RBC ||
				  (Common.ComponentClass) _bloodUnit.BloodComponentClassId == Common.ComponentClass.WB) &&
				_bloodUnit.BloodUnitMedia.AboRh.RH == Common.RH.Positive && _patient.AboRh.RH == Common.RH.Negative)
			{
				_message.Append(Common.StrRes.ValidMsg.UC003.BR321().ResString + "\n");
			}

		
			// CR2899: start
			if (_emergencyIssue && _bloodUnit.EligibleForEmergencyIssue(
				_orderedUnit.CrossmatchRequiredIndicator, 
				_patient.ABORhSource == Common.ABORhSource.ConversionDiscrepancy ||
				_patientABORhSourceCache == Common.ABORhSource.SpecimenTestDiscrepancy || 
				_orderedComponent.SpecimenIdentified && _orderedComponent.GetSpecimen().RepeatABORhTestsMatch == Common.TypingTestResults.Discrepancy,  // CR3114
				_orderedComponent.SpecimenIdentified && _orderedComponent.GetSpecimen().ABORhTested))  // CR3114
			{
				_message.Append(Common.StrRes.InfoMsg.UC043.EmergencyIssue().ResString);
			}
			// CR2899: end

		

			#endregion
		}

		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/28/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5896"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>Bool</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5897"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Returns a boolean indicating whether or not the unit must be emergency issued
		/// </summary>
		/// <returns></returns>
		public bool IsEmergencyIssue()
		{
			// When specimen required, specimen must be present and fully tested.
			// CR2899 start: OrderedComponent doesn't need to be recreated
			if ( _orderedComponent.SpecimenRequiredIndicator )
			{
				if (_orderedComponent.SpecimenIdentified == false)  // CR3114
				{
					return true;
				}
				else
				{
					if (!_orderedComponent.GetSpecimen().TASTested)  // CR3114
					{
						return true;
					}
				}
			}
			// CR2899 end


			// BR_3.05: Patient must have specimen test results and ABO/Rh may not 
			// be inconclusive
			//CR2801 Removed ConversionDiscrepancy check
			// CR2899: get patient ABO/Rh from cache object
			if (_patientABORhSourceCache == Common.ABORhSource.Conversion
				|| _patientABORhSourceCache == Common.ABORhSource.None
				|| _patient.AboRh.Abo == Common.ABO.I 
				|| _patient.AboRh.RH == Common.RH.Inconclusive)
			{
				return true;
			}


			Common.CrossmatchResult crossmatchResult = Common.Utility.GetCrossmatchResultEnumFromString( _orderedUnit.CurrentCrossmatchResultCode );
			if (_orderedUnit.CrossmatchRequiredIndicator == true && 
				( crossmatchResult != Common.CrossmatchResult.CompatibleEXM &&
				crossmatchResult != Common.CrossmatchResult.CompatibleSafeToTransfuse &&
				crossmatchResult != Common.CrossmatchResult.IncompatibleDirectorApprovalNeeded) )
			{
				return true;
			}


			// BR_3.30, BR_3.40
			if (_orderedComponent.SpecimenIdentified && (  // CR3114
				//CR2801 start
				_patient.ABORhSource == Common.ABORhSource.ConversionDiscrepancy ||
				_patient.ABORhSource == Common.ABORhSource.SpecimenTestDiscrepancy ||
				_orderedComponent.GetSpecimen().RepeatABORhTestsMatch == Common.TypingTestResults.Discrepancy))  // CR3114
				//CR2801 end
			{
				return true;
			}
			
			return false;
		}


		///<summary>
		/// Verifies that the user has authority to issue unit.
		///</summary>
		private bool HasAuthority(Common.FunctionCode function)
		{			
			return _user.IsAuthorizedForFunction( function );
		}


		#region Zebra

		/// <summary>
		/// Formats a ZPL (Zebra Programming Language) set of commands.  These commands represent
		/// the native language of the Zebra printer.
		/// BR_43.03, BR_43.04, BR_43.08, BR_43.10
		/// </summary>
		/// <returns></returns>
		private string FormatZPL()
		{
			Division div = new Division(Common.LogonUser.LogonUserDivisionCode);
			string crossmatch	= string.Empty;
			string bloodProduct = string.Empty;
			string aboRh		= string.Empty;
			string emergencyIssueText = string.Empty;
			string techs		= string.Empty;

			if (_bloodUnit.BarcodeType == Common.BarcodeType.ISBT128)
			{
				bloodProduct = _bloodUnit.ProductCode.Trim() + _bloodUnit.DonationType.DonationTypeCode.ToString().Trim() + _bloodUnit.UnitDivisionId;
			}
			else
			{
				bloodProduct = _bloodUnit.ProductCode.Trim();
			}
			
			techs = new BOL.VbecsUser(_orderedUnit.SelectedTechId).UserName;
			if (_orderedUnit.CrossmatchRequiredIndicator)
			{
				// BR_43.04
				if (_orderedComponent.SpecimenIdentified == false || !_orderedComponent.GetSpecimen().TASTested) // CR3114
				{
					crossmatch = Common.StrRes.OtherMsg.UC043.NoCrossmatchAtIssue().ResString;
				}
				else
				{
					// BR_43.03, BR_43.04
					crossmatch = Common.Utility.GetCautionTagCrossmatchText(Common.Utility.GetCrossmatchResultEnumFromString(_orderedUnit.CurrentCrossmatchResultCode));
					if (_orderedUnit.CurrentCrossmatchDate != DateTime.MinValue)
					{
						crossmatch = crossmatch + ", " + Common.VBECSDateTime.FormatDateTimeString(_orderedUnit.CurrentCrossmatchDate);
						// This doesn't seem right, but this condition is happening in the DB.
						if (_orderedUnit.CurrentCrossmatchTechId != string.Empty)
						{
							techs = techs + "/ " + new BOL.VbecsUser(_orderedUnit.CurrentCrossmatchTechId).UserName; 
						}
					}
				}
			}
			else
			{
				if (_emergencyIssue)
				{
					crossmatch = Common.StrRes.OtherMsg.UC043.NotRequired().ResString;
				}
				else
				{
					crossmatch = Common.StrRes.OtherMsg.UC043.NotApplicable().ResString;
				}
			}

			// BR_43.04
			if (!_orderedComponent.SpecimenRequiredIndicator ||
				(_orderedComponent.SpecimenIdentified && _orderedComponent.GetSpecimen().ABORhTested)) // CR3114
			{
				aboRh = _patient.AboRh.AboRHText;
			}
			else
			{
				aboRh = "Unknown";
			}


			string formattedUnit = GetFormattedUnit(_bloodUnit);
			
			
		

			
			if (_emergencyIssue) emergencyIssueText = "^FO480,760^A0N,92,57^FD**EMERGENCY ISSUE**^FS";
	
			// XA:	start of ZPL document
			// B7:	PDF417 barcode
			// LH:  Label Home
			// FO:  Field Origin relative to Label Home (LH)
			// FD:  Field Data
			// FS:  Field Separator, denotes the end of a field definition

			// CR2454: Various ZPL changes to accomodate a 30 character patient name
			string zplString = 
				"^XA^LH70,30" + 

				
				//Barcode
				"^FO625,5,^B7R,26,Y,N,N" +

				// Patient ID (Barcode) - BR_43.08, BR_43.10
				"^FD" + _patient.DisplayPatientID.Trim() + "," +
				// Product code (Barcode) - BR_43.08, BR_43.10
				bloodProduct + "," + 
				// Unit ID (Barcode) - BR_43.08, BR_43.10
				_bloodUnit.EyeReadableUnitId.Trim() + "^FS" +
		
				

				//Product Code: added per CR3064
				"^FO20,40^A0N,32,25" + 
				"^FDProduct Code: " +
				"^FS^FO20,90^A0N,70,58" + 
				"^FD" + _bloodUnit.FullProductCode + 


				"^FS^FO20,250^A0N,56,39" + 


				// Division name - BR_43.08
				"^FD" + div.DivisionName  + "," +
		
				"^FS^FO20,300^A0N,56,39" +
	
		
				// Address - BR_43.08
				"^FD" + div.Address.City + ", " + div.Address.StateCode + "^FS" +

				"^FO1,410^GB1,725,4" +
				"^FS^FO1,410^GB1080,725,4" +
				"^FS^FO1,410^GB1080,0,4^FS^FO20,420^A0N,32,25" + 
				
				"^FDPatient Name:" + "^FS" + GetPatientZPL(_patient.DisplayName) + 			
				
				
				"^FS^FO1,510^GB1080,0,4^FS^FO20,520^A0N,32,25" + 
				
				// Patient ID
				"^FDPatient ID:^FS^FO165,530^A0N,75,62" +
				"^FD" + _patient.DisplayPatientID + "^FS^FO660,510^GB0,110,4^FS" +
				
				"^FO690,520^A0N,32,25^FDRecipient Blood Type:^FS" + 
				"^FO690,560^A0N,56,39" +

				// Patient ABO/Rh - BR_43.08
				"^FD" + aboRh + "^FS" +
				//
				"^FO1,620^GB1080,0,4^FS" +
				"^FO20,630^A0N,32,25^FDUnit/Pool ID:^FS" + 

				// Unit ID
				"^FO170,640^A0N,75,62^FD" + formattedUnit + "^FS" +


				"^FO660,620^GB0,115,4^FS" + 
				"^FO690,630^A0N,32,25^FDDonor Blood Type:^FS" +
				// Unit ABO/Rh - BR_43.08
				"^FO690,670^A0N,56,39^FD" + _bloodUnit.BloodUnitMedia.AboRh.AboRHText + "^FS" + 
				"^FO1,730^GB1080,0,4^FS" +
				"^FO20,760^A0N,32,25^FDCompatibility Interpretation:^FS" +
				// Compatibility - BR_43.08
				"^FO20,800^A0N,56,39^FD" + crossmatch + "^FS" +
				// Emergency Issue
				emergencyIssueText +

				"^FO1,860^GB1080,0,4^FS" + 
				// Caution statement - BR_43.08
				"^FO20,870^A0N,40,30^FDCAUTION: Before transfusing this blood component ^FS" + 
				"^FO640,870^A0N,40,30^FDperform bedside^FS" +
				"^FO20,915^A0N,40,30^FDverification of the patient's identity by ^FS" +
				"^FO485,915^A0N,40,30^FDmatching the patient identification on this^FS" +
				"^FO20,960^A0N,40,30^FDtag with the patient wristband and the Blood ^FS" +
				"^FO560,960^A0N,40,30^FDTransfusion Record Form.^FS" +
				"^FO1,1030^GB1080,0,4^FS" +
				"^FO20,1040^A0N,32,25^FDAssigned Date/Time:^FS" +
				// Selection date - BR_43.08
				"^FO20,1080^A0N,56,39^FD" + Common.VBECSDateTime.FormatDateTimeString(_orderedUnit.SelectedDate) + "^FS" +
				"^FO430,1030^GB0,125,4^FS" +
				"^FO450,1040^A0N,32,25^FDAssigning Tech/ Crossmatch Tech:^FS" +
				// Tech ID - BR_43.08
				"^FO450,1080^A0N,48,31^FD" + techs + "^FS" +
				"^FO1,1155^GB1080,0,4^FS" +
				"^XZ";


			return zplString; 
		}


		private string GetFormattedUnit(BOL.BloodUnit unit)
		{
			// Add spaces in the unit ID field per DR3752
			if (unit.BarcodeType == Common.BarcodeType.ISBT128)
			{
				return unit.EyeReadableUnitId.Substring(0,5) + " " + 
					unit.EyeReadableUnitId.Substring(5, 2) + " " +
					unit.EyeReadableUnitId.Substring(7);
			}
			else
			{
				return unit.EyeReadableUnitId;
			}
		}

		/// <summary>
		/// This updates the ZPL that represents the patient name based on its length. If the name
		/// is longer, the font is made smaller so it will fit. This was implemented to account
		/// for tag layout changes caused by DR1704.
		/// </summary>
		/// <param name="name"></param>
		/// <returns></returns>
		private string GetPatientZPL(string name)
		{		
			if (name.Length > 60)
			{
				return "^FO20,450^A0N,25,18^FD" + name;
			}
			else if (name.Length > 45)
			{
				return "^FO20,450^A0N,40,25^FD" + name;
			}
			else if (name.Length > 30)
			{
				return "^FO20,450^A0N,52,34^FD" + name;
			}
			else if (name.Length > 20)
			{
				return "^FO20,450^A0N,56,39^FD" + name;
			}
			else
			{
				return "^FO180,430^A0N,75,62^FD" + name;
			}
		}


		/// <summary>
		/// Sends the command to the printer via TCP/IP
		/// </summary>
		/// <param name="zpl"></param>
		/// <param name="backupTags"></param>
		/// <returns></returns>
		private bool Print(string zpl, bool backupTags)
		{
			try
			{
				//Save Exception Report 
				if (!GenerateExceptionReport()) return false;
				
				if (zpl == null || zpl.Trim().Length==0)
				{
					throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat("ZPL").ResString);
				}
		        //Defect 227411
				// Create the Socket and open it.
                using (TcpClient tcpClient = new TcpClient())
                {

                    if (_division.LabelPrinterConfig == null || _division.LabelPrinterConfig.ListenerIPAddress == null ||
                        _division.LabelPrinterConfig.ListenerTcpPortNumber < 1 || _division.LabelPrinterConfig.ListenerTcpPortNumber > 50000)
                    {
                        return false;
                    }
                    else
                    {
                        tcpClient.Connect(_division.LabelPrinterConfig.ListenerIPAddress, _division.LabelPrinterConfig.ListenerTcpPortNumber);

                        String str = Console.ReadLine();
                        Stream stm = tcpClient.GetStream();

                        ASCIIEncoding asEncoding = new ASCIIEncoding();
                        byte[] byteArray = asEncoding.GetBytes(zpl);

                        stm.Write(byteArray, 0, byteArray.Length);
                        stm.Flush();
                        stm.Close();
                        tcpClient.Close();
                        return true;
                    }
                }
			}
			catch (System.Net.Sockets.SocketException se)
			{
				if (!backupTags) throw se;
				Console.WriteLine("Connection failed");
				return false;
			}
		}

		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/8/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7432"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>true</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="7433"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>false</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// GenerateExceptionReport
		/// </summary>
		/// <returns></returns>
		public bool GenerateExceptionReport()
		{
			if (!_exceptionReportGenerated && _exceptionReportEntries != null && _exceptionReportEntries.Count > 0)
			{
				DataTable dtExceptionReport = null;
				foreach(BOL.ExceptionReport eR in _exceptionReportEntries)
				{
					if (dtExceptionReport == null) dtExceptionReport = eR.ExceptionData.Table.Clone();

					dtExceptionReport.Rows.Add(eR.ExceptionData.ItemArray);
				}

				_exceptionReportGenerated = true;
				if (dtExceptionReport != null)
				{
					return DAL.CautionTag.InsertExceptionReport(dtExceptionReport) == 0;
				}
				else
				{
					return true;
				}
			}
			else
			{
				return true;
			}
		}

		#endregion


		#region overridden

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/8/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7434"> 
		///		<ExpectedInput>DataRow</ExpectedInput>
		///		<ExpectedOutput>BusinessObjectException</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="7435"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// LoadDataRowFromThis
		/// </summary>
		/// <param name="dtRow"></param>
		/// <returns></returns>
		public override DataRow LoadDataRowFromThis(DataRow dtRow)
		{
			throw new BOL.BusinessObjectException("This has not been implemented!");
		}

		/// <summary>
		/// LoadFromDataRow
		/// </summary>
		/// <param name="dtRow"></param>
		protected override void LoadFromDataRow(System.Data.DataRow dtRow)
		{
			throw new BOL.BusinessObjectException("This has not been implemented!");
		}

		#endregion

		#endregion

	}
}
